<?php
require_once('SSHCommand.php');
require_once('CommonUtility.class.php');

class GatewayInfo
{
    const TYPE_STATIC = "static";
    const TYPE_DHCP = "dhcp";
    const TYPE_ISOLATED = "isolated";
    const TYPE_NAT = 'nat';

    protected $_type = null;
    protected $_ip = null;
    protected $_mask = null;
    protected $_gateway = null;
    protected $_dns_list = array();

    function __construct($type, $ip, $mask, $gateway, $dns_list)
    {
        $this->_type = $type;
        $this->_ip = $ip;
        $this->_mask = $mask;
        $this->_gateway = $gateway;
        $this->_dns_list = $dns_list;
    }

    public function isValid()
    {
        if (empty($this->_dns_list)) return false;
        $ip_valid_list = array_merge(array($this->_ip, $this->_mask, $this->_gateway), $this->_dns_list);
        foreach ($ip_valid_list as $ip) {
            if (!filter_var($ip, FILTER_VALIDATE_IP)) return false;
        }
        return true;
    }

    public function getFirstDns()
    {
        if (empty($this->_dns_list)) return null;
        return reset($this->_dns_list);
    }

    public function getDnsList()
    {
        return $this->_dns_list;
    }

    public function getGateway()
    {
        return $this->_gateway;
    }

    public function getIP()
    {
        return $this->_ip;
    }

    public function getMask()
    {
        return $this->_mask;
    }

    public function getType()
    {
        return $this->_type;
    }

    public function isDHCP()
    {
        return $this->_type == GatewayInfo::TYPE_DHCP;
    }

    public function isStatic()
    {
        return $this->_type == GatewayInfo::TYPE_STATIC;
    }

    public function isIsolated()
    {
        return $this->_type == GatewayInfo::TYPE_ISOLATED;
    }
}

class InvalidGatewayInfo
{
    public function isValid()
    {
        return false;
    }
}

class ClusterConfigRetriever
{
    protected $_cfg_xml = null;
    protected $_cfg_xml_path = null;

    function __construct($cfg_xml_path)
    {
        $this->_cfg_xml_path = $cfg_xml_path;
        $this->_cfg_xml = simplexml_load_file($cfg_xml_path);
    }

    public function getClusterConfig($sc_id, $cluster_cfg_path)
    {
        $cluster_section = "cluster_$sc_id";
        $esxi_ip = (string)$this->_cfg_xml->$cluster_section->esxi->ip;
        $esxi_user = (string)$this->_cfg_xml->$cluster_section->esxi->username;
        $esxi_pwd = (string)$this->_cfg_xml->$cluster_section->esxi->password;
        $ms_prefix = (string)$this->_cfg_xml->$cluster_section->management_server->prefix;
        $cluster_sc_vm_path = get_vm_vixpath($esxi_ip, $esxi_user, $esxi_pwd, $ms_prefix);
        $cluster_copy_cfg_info = $this->getClusterCopyCfgInfo($esxi_ip, $esxi_user, $esxi_pwd, $cluster_sc_vm_path, $cluster_cfg_path);
        copy_file_from_ms($cluster_copy_cfg_info);
        if (file_exists($cluster_cfg_path)) return;
        $mask_cluster_copy_cfg_info = CommonUtility::mask_confidential_list($cluster_copy_cfg_info, array('esxi_username', 'esxi_password'));
        debug_print("Cannot copy cluster's dtascfg.xml. where cluster's info is " . var_export($mask_cluster_copy_cfg_info, true));
        throw new Exception("Cannot get cluster's config");
    }

    protected function getClusterCopyCfgInfo($esxi_ip, $esxi_user, $esxi_pwd, $cluster_sc_vm_path, $cluster_cfg_path)
    {
        $cluster_copy_cfg_info = array(
            'esxi_ip' => $esxi_ip,
            'esxi_username' => $esxi_user,
            'esxi_password' => $esxi_pwd,
            'ms_vmpath' => $cluster_sc_vm_path,
            'from' => $this->_cfg_xml_path,
            'to' => $cluster_cfg_path,
        );
        return $cluster_copy_cfg_info;
    }
}

class GatewayRetriever
{
    protected $_cfg_xml = null;
    protected $_usbx_cli_path = null;
    protected $_ssh_cmd_retriever = null;

    function __construct($cfg_xml_path, $is_internal_sc = false)
    {
        if ($is_internal_sc) {
            $this->_ssh_cmd_retriever = new InternalSandboxControllerSSHCmdRetriever($cfg_xml_path);
        }
        else {
            $this->_ssh_cmd_retriever = new SandboxControllerSSHCmdRetriever($cfg_xml_path);
        }
        $this->_cfg_xml = simplexml_load_file($cfg_xml_path);
        $this->_usbx_cli_path = (string)$this->_cfg_xml->u_sandbox->cli_path;
    }

    protected function getGatewayType($is_dhcp, $is_isolated)
    {
        if ($is_isolated) return GatewayInfo::TYPE_ISOLATED;
        if ($is_dhcp) return GatewayInfo::TYPE_DHCP;
        return GatewayInfo::TYPE_STATIC;
    }

    public function getGatewayInfo()
    {
        $is_dhcp = (bool)(int)$this->_cfg_xml->nat->dhcp;
        $is_isolated = !(bool)(int)$this->_cfg_xml->nat->internet_connection;
        $ip = (string)$this->_cfg_xml->nat->ip;
        $mask = (string)$this->_cfg_xml->nat->netmask;
        $dns = (string)$this->_cfg_xml->nat->dns;
        $gateway = (string)$this->_cfg_xml->nat->gateway;
        $type = $this->getGatewayType($is_dhcp, $is_isolated);
        return new GatewayInfo($type, $ip, $mask, $gateway, array($dns));
    }

    public function getGateway()
    {
        $ssh_cmd = $this->_ssh_cmd_retriever->getSSHCmd();
        $gateway = new UsbxGateway($this->_usbx_cli_path, $ssh_cmd);
        return $gateway;
    }

}

class UsbxGateway
{
    const GATEWAY_RES_TEMPLATE = <<<heredoc
Type: (?P<type>.*)
NIC: .*
IP: (?P<ip>.*)
Netmask: (?P<mask>.*)
DNS: (?P<dns>.*)
Gateway: (?P<gateway>.*)
heredoc;

    protected $_usbx_cli_path = null;
    protected $_ssh_cmd = null;
    protected $_def_gateway_cmd_list = null;

    function __construct($usbx_cli_path, $ssh_cmd)
    {
        $this->_usbx_cli_path = $usbx_cli_path;
        $this->_ssh_cmd = $ssh_cmd;
        $this->_def_gateway_cmd_list = array($this->_usbx_cli_path, 'set-gateway');
    }

    public function setGatewayInfo(GatewayInfo $gateway_info)
    {
        if ($gateway_info->isIsolated()) return $this->setIsolated();
        if ($gateway_info->isDHCP()) return $this->setDHCP();
        $ip = $gateway_info->getIP();
        $mask = $gateway_info->getMask();
        $gateway = $gateway_info->getGateway();
        $dns = $gateway_info->getFirstDns();
        return $this->setStatic($ip, $mask, $gateway, $dns);
    }

    public function setIsolated()
    {
        $set_isolated_cmd_list = $this->getIsolatedCmdList();
        $gw_info = $this->doGatewayCmd($set_isolated_cmd_list);
        return $gw_info->isIsolated();
    }

    public function setStatic($ip, $mask, $gateway, $dns)
    {
        $set_static_cmd_list = $this->getStaticCmdList($ip, $mask, $gateway, $dns);
        $gw_info = $this->doGatewayCmd($set_static_cmd_list);
        return $gw_info->isStatic();
    }

    protected function getIsolatedCmdList()
    {
        $cmd_list = array('--type', 'isolated');
        return $cmd_list;
    }

    protected function getStaticCmdList($ip, $mask, $gateway, $dns)
    {
        $cmd_list = array('--type', 'static', '--bindif', 'eth2',
            '--ip', $ip, '--netmask', $mask, '--gateway', $gateway, '--dns', $dns);
        return $cmd_list;
    }


    public function setDHCP($timeout_in_sec = 120)
    {
        $set_dhcp_cmd_list = $this->getDHCPCmdList();
        $this->doGatewayCmd($set_dhcp_cmd_list);
        $start_time = time();
        while (time() - $start_time < $timeout_in_sec) {
            $gw_info = $this->getGatewayInfo();
            if ($gw_info->isValid()) return true;
        }
        return false;
    }

    protected function getDHCPCmdList()
    {
        $cmd_list = array('--type', 'dhcp', '--bindif', 'eth2');
        return $cmd_list;
    }

    protected function parseGatewayResult($result)
    {
        preg_match("/" . UsbxGateway::GATEWAY_RES_TEMPLATE . "/", $result, $matches);
        if (!$matches) {
            debug_print("Cannot match result $result from gateway");
            return new InvalidGatewayInfo();
        }
        $type = $matches['type'];
        $ip = $matches['ip'];
        $mask = $matches['mask'];
        $gateway = $matches['gateway'];
        $dns_list = explode(",", $matches['dns']);
        return new GatewayInfo($type, $ip, $mask, $gateway, $dns_list);
    }

    protected function doGatewayCmd($cmd_list = array())
    {
        $gateway_cmd_list = array_merge($this->_def_gateway_cmd_list, $cmd_list);
        $gateway_cmd = implode(" ", $gateway_cmd_list);
        $out = $this->_ssh_cmd->doCmd($gateway_cmd);
        $gateway_info = $this->parseGatewayResult($out);
        return $gateway_info;
    }

    public function getGatewayInfo()
    {
        return $this->doGatewayCmd();
    }

    public function isCompatible($usbx_gateway)
    {
        $gateway_info = $this->getGatewayInfo();
        $is_gateway_isolated = $gateway_info->isIsolated();
        $compared_gateway_info = $usbx_gateway->getGatewayInfo();
        $is_compared_gateway_isolated = $compared_gateway_info->isIsolated();
        return $is_gateway_isolated == $is_compared_gateway_isolated;
    }
}
